REAL FUNCTION corponegro(comp_onda)
    !Função de corpo negro para um dado comprimento de onda (comp_onda) e temperatura T = 2500K
    
    IMPLICIT NONE
    
    REAL, INTENT(IN) :: comp_onda !Comprimento de onda a ser calculada a função
    REAL :: expoente, divisor, k, h, c, pi !Partes do cálculo da função e constantes
    
    !Constantes em CGS (constante de boltzmann, de planck e da velocidade da luz no vácuo) e uma aproximação para pi
    k = 1.381E-16
    h = 6.626E-27
    c = 2.998E10
    pi = 3.142
                        
    !Calculando a função
    expoente = (h * c / (comp_onda * k * 2500.)) !2500K = T
    divisor = comp_onda**5. * (exp(expoente) - 1.)
        
    corponegro = (2. * h * c**2.) / divisor
    
END FUNCTION corponegro

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

REAL FUNCTION trapezio(x0, x1, n)
    !Calcula a integral da função de corpo negro em um intervalo [x0, x1] dividido em n subintervalos

    IMPLICIT NONE
    
    REAL, INTENT(IN) :: x0, x1 !Limites do intervalo
    REAL, SAVE :: anterior !Estimativa anterior calculada (salva a cada chamada da função)
    REAL, EXTERNAL :: corponegro !Função a ser integrada
    REAL :: somatoria, termo, h !Variáveis auxiliares, tamanho h dos subintervalos
    INTEGER :: n !Quantidade n de subintervalos
    
    !Definindo h, termo e valor inicial da somatoria
    h = (x1 - x0) / n
    termo = x0 + h
    somatoria = 0.
    
    !Caso haja apenas 1 intervalo
    IF (n .EQ. 1) THEN
    
        trapezio = h * (corponegro(x0) + corponegro(x1)) / 2
       
    !No caso de mais intervalos
    ELSE
    
        DO WHILE (termo < x1)
        
            somatoria = somatoria + corponegro(termo)            
            termo = termo + 2 * h
            
        END DO
        
        !Cálculo da integral
        trapezio = anterior / 2 + h * somatoria
        
    END IF
    
    !Atualizando o valor a ser salvo
    anterior = trapezio
    
END FUNCTION trapezio

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

PROGRAM integracao_trapezio

    IMPLICIT NONE
    
    REAL, DIMENSION(8) :: eps !Array com as precisões desejadas
    REAL :: lbd0, lbd1, eps_integral, integral0, integral1 !Extremos do intervalo de comprimento de onda, erro relativo de cada passo, estimativas anterior e atual da integral
    INTEGER :: n, i !Número n de intervalos, variável auxiliar
    REAL, EXTERNAL :: trapezio !Função que realiza a integração
    
    !Definindo o intervalo
    lbd0 = 5.E-5 !0.5 microns -> cm
    lbd1 = 1.E-3 !10 microns -> cm
    
    !Definindo o array eps
    eps = (/1.E-3, 1.E-4, 1.E-5, 1.E-6, 1.E-7, 1.E-8, 1.E-9, 1.E-10/)
    
    !Calculando as integrais para cada valor de eps
    n = 1
    integral0 = 0.
    integral1 = trapezio(lbd0, lbd1, n)
    eps_integral = ABS((integral1 - integral0) / integral1)
    
    PRINT *, "As colunas abaixo correspondem, respectivamente a: epsilon, número de passos e valor da integral"
    
    DO i = 1, 8
    
        DO WHILE (eps_integral >= eps(i) .AND. n < 2**22) !Condição limitando n para evitar "loops infinitos"

            n = n * 2
            integral0 = integral1
            integral1 = trapezio(lbd0, lbd1, n)
            eps_integral = ABS((integral1 - integral0) / integral1)
        
        END DO
        
        PRINT *, eps(i), n, integral1
    
    END DO

END PROGRAM integracao_trapezio
